/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import org.eclipse.lsp4j.jsonrpc.messages.Either;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;

/**
 * A document filter denotes a document through properties like language, schema or pattern.
 * <p>
 * At least one of the properties {@link #language}, {@link #scheme}, or {@link #pattern} must be set.
 */
@SuppressWarnings("all")
public class DocumentFilter {
  /**
   * A language id, like {@code typescript}.
   */
  private String language;

  /**
   * A uri scheme, like {@code file} or {@code untitled}.
   */
  private String scheme;

  /**
   * A glob pattern, like <code>*.{ts,js}</code>.
   * <p>
   * Since 3.18 - support for relative patterns, which depends on the
   * client capability {@link FiltersCapabilities#relativePatternSupport}.
   */
  private Either<String, RelativePattern> pattern;

  public DocumentFilter() {
  }

  public DocumentFilter(final String language, final String scheme, final Either<String, RelativePattern> pattern) {
    this.language = language;
    this.scheme = scheme;
    this.pattern = pattern;
  }

  /**
   * A language id, like {@code typescript}.
   */
  public String getLanguage() {
    return this.language;
  }

  /**
   * A language id, like {@code typescript}.
   */
  public void setLanguage(final String language) {
    this.language = language;
  }

  /**
   * A uri scheme, like {@code file} or {@code untitled}.
   */
  public String getScheme() {
    return this.scheme;
  }

  /**
   * A uri scheme, like {@code file} or {@code untitled}.
   */
  public void setScheme(final String scheme) {
    this.scheme = scheme;
  }

  /**
   * A glob pattern, like <code>*.{ts,js}</code>.
   * <p>
   * Since 3.18 - support for relative patterns, which depends on the
   * client capability {@link FiltersCapabilities#relativePatternSupport}.
   */
  public Either<String, RelativePattern> getPattern() {
    return this.pattern;
  }

  /**
   * A glob pattern, like <code>*.{ts,js}</code>.
   * <p>
   * Since 3.18 - support for relative patterns, which depends on the
   * client capability {@link FiltersCapabilities#relativePatternSupport}.
   */
  public void setPattern(final Either<String, RelativePattern> pattern) {
    this.pattern = pattern;
  }

  public void setPattern(final String pattern) {
    if (pattern == null) {
      this.pattern = null;
      return;
    }
    this.pattern = Either.forLeft(pattern);
  }

  public void setPattern(final RelativePattern pattern) {
    if (pattern == null) {
      this.pattern = null;
      return;
    }
    this.pattern = Either.forRight(pattern);
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("language", this.language);
    b.add("scheme", this.scheme);
    b.add("pattern", this.pattern);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    DocumentFilter other = (DocumentFilter) obj;
    if (this.language == null) {
      if (other.language != null)
        return false;
    } else if (!this.language.equals(other.language))
      return false;
    if (this.scheme == null) {
      if (other.scheme != null)
        return false;
    } else if (!this.scheme.equals(other.scheme))
      return false;
    if (this.pattern == null) {
      if (other.pattern != null)
        return false;
    } else if (!this.pattern.equals(other.pattern))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.language== null) ? 0 : this.language.hashCode());
    result = prime * result + ((this.scheme== null) ? 0 : this.scheme.hashCode());
    return prime * result + ((this.pattern== null) ? 0 : this.pattern.hashCode());
  }
}
