package org.eclipse.nebula.visualization.xygraph.examples;

/*******************************************************************************
 * Copyright (c) 2010 Oak Ridge National Laboratory.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 ******************************************************************************/
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.KeyEvent;
import org.eclipse.draw2d.KeyListener;
import org.eclipse.draw2d.LightweightSystem;
import org.eclipse.draw2d.MouseEvent;
import org.eclipse.draw2d.MouseListener;
import org.eclipse.draw2d.TreeSearch;
import org.eclipse.nebula.visualization.xygraph.dataprovider.CircularBufferDataProvider;
import org.eclipse.nebula.visualization.xygraph.dataprovider.Sample;
import org.eclipse.nebula.visualization.xygraph.figures.Axis;
import org.eclipse.nebula.visualization.xygraph.figures.IXYGraph;
import org.eclipse.nebula.visualization.xygraph.figures.PlotArea;
import org.eclipse.nebula.visualization.xygraph.figures.ToolbarArmedXYGraph;
import org.eclipse.nebula.visualization.xygraph.figures.Trace;
import org.eclipse.nebula.visualization.xygraph.figures.Trace.BaseLine;
import org.eclipse.nebula.visualization.xygraph.figures.Trace.ErrorBarType;
import org.eclipse.nebula.visualization.xygraph.figures.Trace.PointStyle;
import org.eclipse.nebula.visualization.xygraph.figures.Trace.TraceType;
import org.eclipse.nebula.visualization.xygraph.figures.XYGraph;
import org.eclipse.nebula.visualization.xygraph.figures.ZoomType;
import org.eclipse.nebula.visualization.xygraph.linearscale.AbstractScale.LabelSide;
import org.eclipse.nebula.visualization.xygraph.linearscale.Range;
import org.eclipse.nebula.visualization.xygraph.util.XYGraphMediaFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseWheelListener;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.ImageLoader;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;

/**
 * A Comprehensive Example
 * 
 * @author Xihui Chen
 */
public class ComprehensiveExample {
	public static void main(final String[] args) {
		final Shell shell = new Shell();
		shell.setSize(800, 500);
		shell.open();

		final LightweightSystem lws = new LightweightSystem(shell);
		final ToolbarArmedXYGraph toolbarArmedXYGraph = createXYGraph();
		lws.setContents(toolbarArmedXYGraph);
		shell.addMouseWheelListener(new MouseWheelListener() {

			@Override
			public void mouseScrolled(org.eclipse.swt.events.MouseEvent e) {
				IFigure figureUnderMouse = toolbarArmedXYGraph.findFigureAt(e.x, e.y, new TreeSearch() {

					@Override
					public boolean prune(IFigure figure) {
						return false;
					}

					@Override
					public boolean accept(IFigure figure) {
						return figure instanceof Axis || figure instanceof PlotArea;
					}
				});
				if (figureUnderMouse instanceof Axis) {
					Axis axis = ((Axis) figureUnderMouse);
					double valuePosition = axis.getPositionValue(axis.isHorizontal() ? e.x : e.y, false);
					axis.zoomInOut(valuePosition, e.count * 0.1 / 3);
				} else if (figureUnderMouse instanceof PlotArea) {
					PlotArea plotArea = (PlotArea) figureUnderMouse;
					plotArea.zoomInOut(true, true, e.x, e.y, e.count * 0.1 / 3);
				}
			}

		});
		shell.setText("Comprehensive Example");
		final Display display = Display.getDefault();
		while (!shell.isDisposed()) {
			if (!display.readAndDispatch()) {
				display.sleep();
			}
		}
	}

	private static ToolbarArmedXYGraph createXYGraph() {
		final Trace trace2;
		final Trace trace3;
		final IXYGraph xyGraph;
		Runnable updater;
		final CircularBufferDataProvider trace2Provider;
		final Trace trace4;
		final ToolbarArmedXYGraph toolbarArmedXYGraph;
		xyGraph = new XYGraph();
		xyGraph.setTitle("XY Graph Test");
		xyGraph.setFont(XYGraphMediaFactory.getInstance().getFont(XYGraphMediaFactory.FONT_TAHOMA));
		xyGraph.getPrimaryXAxis().setTitle("Time");
		xyGraph.getPrimaryYAxis().setTitle("Amplitude");
		xyGraph.getPrimaryXAxis().setRange(new Range(0, 200));
		xyGraph.getPrimaryXAxis().setDateEnabled(true);
		xyGraph.getPrimaryYAxis().setAutoScale(true);
		xyGraph.getPrimaryXAxis().setAutoScale(true);
		xyGraph.getPrimaryXAxis().setShowMajorGrid(true);
		xyGraph.getPrimaryYAxis().setShowMajorGrid(true);
		xyGraph.getPrimaryXAxis().setAutoScaleThreshold(0);

		final Axis x2Axis = new Axis("X-2", false);
		x2Axis.setTickLabelSide(LabelSide.Secondary);
		// x2Axis.setAutoScale(true);
		xyGraph.addAxis(x2Axis);

		final Axis y2Axis = new Axis("Log Scale", true);
		y2Axis.setRange(10, 500);
		y2Axis.setLogScale(true);
		// y2Axis.setAutoScale(true);
		y2Axis.setForegroundColor(XYGraphMediaFactory.getInstance().getColor(XYGraphMediaFactory.COLOR_PINK));
		y2Axis.setTickLabelSide(LabelSide.Secondary);
		xyGraph.addAxis(y2Axis);

		Axis y3Axis = new Axis("Y-3", true);
		y3Axis.setForegroundColor(XYGraphMediaFactory.getInstance().getColor(XYGraphMediaFactory.COLOR_BLUE));
		y3Axis.setTickLabelSide(LabelSide.Secondary);
		y3Axis.setRange(new Range(-2, 3));
		y3Axis.setShowMajorGrid(false);
		y3Axis.setAutoScale(true);
		// xyGraph.addAxis(y3Axis);

		trace2Provider = new CircularBufferDataProvider(true);
		trace2Provider.setBufferSize(100);
		trace2Provider.setUpdateDelay(100);

		trace2 = new Trace("Trace 2", xyGraph.getPrimaryXAxis(), xyGraph.getPrimaryYAxis(), trace2Provider);
		trace2.setDataProvider(trace2Provider);
		trace2.setTraceType(TraceType.SOLID_LINE);
		trace2.setLineWidth(1);
		trace2.setPointStyle(PointStyle.POINT);
		trace2.setPointSize(4);
		trace2.setBaseLine(BaseLine.NEGATIVE_INFINITY);
		trace2.setAreaAlpha(100);
		trace2.setAntiAliasing(true);
		trace2.setErrorBarEnabled(false);
		// trace2.setDrawYErrorInArea(true);
		trace2.setYErrorBarType(ErrorBarType.BOTH);
		trace2.setXErrorBarType(ErrorBarType.NONE);
		trace2.setErrorBarCapWidth(3);

		final CircularBufferDataProvider trace3Provider = new CircularBufferDataProvider(true);
		trace3 = new Trace("Trace3", xyGraph.getPrimaryXAxis(), xyGraph.getPrimaryYAxis(), trace3Provider);
		trace3.setPointStyle(PointStyle.XCROSS);
		trace3.setTraceType(TraceType.BAR);
		trace3.setLineWidth(4);
		trace3Provider.setUpdateDelay(100);
		xyGraph.addTrace(trace3);
		xyGraph.addTrace(trace2);

		final CircularBufferDataProvider trace4Provider = new CircularBufferDataProvider(false);
		trace4 = new Trace("Trace 4-Lissajous", x2Axis, y2Axis, trace4Provider);
		// trace4.setPointStyle(PointStyle.POINT);
		trace4.setPointSize(2);

		trace4Provider.setUpdateDelay(100);
		trace4Provider.setBufferSize(100);
		xyGraph.addTrace(trace4);

		toolbarArmedXYGraph = new ToolbarArmedXYGraph(xyGraph);

		// add key listener to XY-Graph. The key pressing will only be monitored
		// when the
		// graph gains focus.

		xyGraph.setFocusTraversable(true);
		xyGraph.setRequestFocusEnabled(true);

		xyGraph.getPlotArea().addMouseListener(new MouseListener.Stub() {
			@Override
			public void mousePressed(final MouseEvent me) {
				xyGraph.requestFocus();
			}

		});

		xyGraph.addKeyListener(new KeyListener.Stub() {
			@Override
			public void keyPressed(final KeyEvent ke) {
				if ((ke.getState() == SWT.CONTROL) && (ke.keycode == 'z')) {
					xyGraph.getOperationsManager().undo();
				}
				if ((ke.getState() == SWT.CONTROL) && (ke.keycode == 'y')) {
					xyGraph.getOperationsManager().redo();
				}
				if ((ke.getState() == SWT.CONTROL) && (ke.keycode == 'x')) {
					xyGraph.performAutoScale();
				}
				if ((ke.getState() == SWT.CONTROL) && (ke.keycode == 's')) {
					final ImageLoader loader = new ImageLoader();
					loader.data = new ImageData[] { xyGraph.getImage().getImageData() };
					final FileDialog dialog = new FileDialog(Display.getDefault().getShells()[0], SWT.SAVE);
					dialog.setFilterNames(new String[] { "PNG Files", "All Files (*.*)" });
					dialog.setFilterExtensions(new String[] { "*.png", "*.*" }); // Windows
					final String path = dialog.open();
					if ((path != null) && !path.equals("")) {
						loader.save(path, SWT.IMAGE_PNG);
					}
				}
				if ((ke.getState() == SWT.CONTROL) && (ke.keycode + 'a' - 97 == 't')) {
					switch (xyGraph.getZoomType()) {
					case RUBBERBAND_ZOOM:
						xyGraph.setZoomType(ZoomType.HORIZONTAL_ZOOM);
						break;
					case HORIZONTAL_ZOOM:
						xyGraph.setZoomType(ZoomType.VERTICAL_ZOOM);
						break;
					case VERTICAL_ZOOM:
						xyGraph.setZoomType(ZoomType.ZOOM_IN);
						break;
					case ZOOM_IN:
						xyGraph.setZoomType(ZoomType.ZOOM_OUT);
						break;
					case ZOOM_OUT:
						xyGraph.setZoomType(ZoomType.PANNING);
						break;
					case PANNING:
						xyGraph.setZoomType(ZoomType.NONE);
						break;
					case NONE:
						xyGraph.setZoomType(ZoomType.RUBBERBAND_ZOOM);
						break;
					default:
						break;
					}
				}
			}
		});

		updater = new Runnable() {
			private double updateIndex;
			private long t = System.currentTimeMillis();
			private boolean running = true;

			public void run() {
				t += 60000;
				trace3Provider.setCurrentYData(Math.cos(updateIndex), t);
				if (((updateIndex >= 10) && (updateIndex <= 10.5)) || ((updateIndex >= 20) && (updateIndex <= 20.2))) {
					trace2Provider.addSample(new Sample(t, Double.NaN));
					running = false;
				} else {
					final Sample sampe = new Sample(t, Math.sin(updateIndex), 0.1 * Math.random(), 0.1 * Math.random(),
							t * 0.0000001 * Math.random(), t * 0.0000001 * Math.random(), "sdfsf");
					trace2Provider.addSample(sampe);
				}

				trace2Provider.setCurrentYDataTimestamp(t);
				trace4Provider.setCurrentXData(Math.sin(updateIndex + 10) * 20 + 50);
				trace4Provider.setCurrentYData(Math.cos(updateIndex) * 30 + 50);
				updateIndex += 0.1;

				if (running) {
					Display.getCurrent().timerExec(1, this);
				}
			}
		};

		Display.getCurrent().timerExec(1000, updater);
		return toolbarArmedXYGraph;
	}
}
